<?php
/**
 * -------------------------------------------------------------------------
 *
 * Модуль для представления, отправляющего данные в мессенджер Telegram.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / Repost Vacancies
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    /**
     * ---------------------------------------------------------------------
     *
     * Подключаем из папки универсальных базовых модулей фреймворка файл
     * "mimimi.modules/Telegram/Telegram.php". Там объявлен базовый класс
     * "MimimiTelegram", который является простым модулем для доступа к
     * Telegram Bot API. Этот класс подходит как основа для реализуемого
     * ниже модуля.
     *
     * ---------------------------------------------------------------------
     */

    mimimiInclude ( 'Telegram/Telegram.php' );

    /**
     * ---------------------------------------------------------------------
     *
     * Создаём на основе того базового класса новый класс, в котором напишем
     * программный код текущего модуля. Обратите внимание как задано имя
     * нового класса - оно сложено из имени класса вышестоящего модуля, то
     * есть "MyMimimiViews", и имени текущего PHP-файла без расширения.
     *
     * ---------------------------------------------------------------------
     */

    class MyMimimiViewsTelegram extends MimimiTelegram {

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Опубликовать вакансию в канале.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   array  $params  (необязательный) Данные от модуля, вызвавшего этот метод.
         *                                           Представляют собой ассоциативный массив из таких элементов:
         *                                               'chat_id' = Идентификатор или имя группы/канала, куда отправляется вакансия.
         *                                               'module'  = Имя модуля, с помощью которого происходил парсинг предоставленной записи.
         *                                               'type'    = Имя типа спарсенных данных, например "vacancies".
         *                                               'row'     = Запись о публикуемой вакансии.
         * @return  int             Идентификатор поста, созданного в Телеграм.
         *
         * -----------------------------------------------------------------
         */

        public function run ( $params = '' ) {
            $token = $this->app->models->settings->get ( 'tg_bot_token' );
            if ( $token ) {
                if ( $this->validateParams ( $params ) ) {
                    $file = 'telegram/' . $params[ 'type' ] . '/' . $params[ 'module' ] . '.tpl';
                    $msg  = $this->generateMessage ( $file, $params[ 'row' ] );
                    $chat = $this->fixChatName ( $params[ 'chat_id' ] );
                    $this->setBotToken ( $token );
                    $result = $this->sendMessage ( $chat, $msg );
                    return    $this->extractId ( $result );
                }
            }
            return 0;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Удалить пост вакансии по её идентификатору.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string|int  $chat_id     Идентификатор или имя группы/канала, в которой удаляется вакансия.
         * @param   int         $message_id  Идентификатор удаляемой вакансии.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function remove ( $chat_id, $message_id ) {
            $token = $this->app->models->settings->get ( 'tg_bot_token' );
            if ( $token ) {
                $chat = $this->fixChatName ( $chat_id );
                $this->setBotToken ( $token );
                $this->deleteMessage ( $chat, $message_id );
            }
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Проверить наличие нужных параметров вызова.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   array  $params  Данные от модуля, вызвавшего этот метод.
         *                          Полностью соответствуют такому же параметру метода run() выше.
         * @return  bool            TRUE если все необходимые параметры присутствуют.
         *
         * -----------------------------------------------------------------
         */

        protected function validateParams ( $params ) {
            return ! empty ( $params[ 'chat_id' ] )
                && ! empty ( $params[ 'module'  ] )
                && ! empty ( $params[ 'type'    ] )
                && ! empty ( $params[ 'row'     ] );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Исправить имя чата, если не соответствует формату.
         *
         * -----------------------------------------------------------------
         */

        protected function fixChatName ( $name ) {
            return preg_replace ( '~^([a-z])~ui', '@$1', $name );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Сгенерировать текст публикации по шаблону.
         *
         * -----------------------------------------------------------------
         */

        protected function generateMessage ( $file, $data ) {
            ob_start ( );
                $this->owner->html->page ([ $file, $data ]);
                $body = ob_get_contents ( );
            ob_end_clean ( );
            return $this->compactMessage ( $body );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Удалить избыточный размер публикации.
         *
         * -----------------------------------------------------------------
         */

        protected function compactMessage ( $msg, $maxsize = 4000 ) {
            $msg = preg_replace ( '~(&nbsp;| |\t)+~ui'             , ' '    , $msg );
            $msg = preg_replace ( '~(\r\n|\n\r|\r)~u'              , "\n"   , $msg );
            $msg = preg_replace ( '~ *\n *~u'                      , "\n"   , $msg );
            $msg = preg_replace ( '~> +~u'                         , '> '   , $msg );
            $msg = preg_replace ( '~\s+</~u'                       , '</'   , $msg );
            $msg = preg_replace ( '~\n\n+~u'                       , "\n\n" , $msg );
            $msg = preg_replace ( '~(^\s+|\s+$)~u'                 , ''     , $msg );
            $msg = preg_replace ( '~\s*<([a-z]+)[^>]*>\s*</\1>~ui' , ''     , $msg );
            $size = mb_strlen ( $msg, 'UTF-8' );
            return $size <= $maxsize ? $msg
                                     : ( mb_substr ( $msg, 0, $maxsize - 1, 'UTF-8' ) . '...' );

        }
    }
