<?php
/**
 * -------------------------------------------------------------------------
 *
 * Модуль для модели Settings.
 *
 * -------------------------------------------------------------------------
 *
 * Эта модель представляет собой таблицу "settings" базы данных. А в классе
 * её модуля, реализованного ниже, имеются следующие методы для работы с
 * этой таблицей:
 *
 *     PUBLIC     onInstall ( $db           )  -->  чтобы произвести необходимые действия при первом запуске приложения
 *     PUBLIC     get       ( $name         )  -->  чтобы получить значение настройки по её имени
 *     PUBLIC     set       ( $name, $value )  -->  чтобы сохранить значение настройки по её имени
 *     PUBLIC     select    (               )  -->  чтобы получить список настроек, позволенных для редактирования
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / Repost Vacancies
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    /**
     * ---------------------------------------------------------------------
     *
     * Подключаем из папки ядра фреймворка файл "mimimi.core/Module.php".
     * Там объявлен класс "MimimiModule", являющийся простейшей модульной
     * заготовкой. Этот класс подходит как основа для реализуемого ниже
     * модуля.
     *
     * ---------------------------------------------------------------------
     */

    mimimiInclude ( 'Module.php' );

    /**
     * ---------------------------------------------------------------------
     *
     * Создаём на основе класса той заготовки новый класс, в котором напишем
     * программный код текущего модуля. Обратите внимание как задано имя
     * нового класса - оно сложено из имени класса вышестоящего модуля, то
     * есть "MyMimimiModels", и имени текущего PHP-файла без расширения.
     *
     * ---------------------------------------------------------------------
     */

    class MyMimimiModelsSettings extends MimimiModule {

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Обработчик первого запуска приложения.
         *
         * -----------------------------------------------------------------
         *
         * Вызов этого метода происходит лишь в ситуации, когда файл SQLite3
         * базы данных ещё не существовал и только что создан. Поэтому здесь
         * мы просто создаём таблицу "settings", описывая желаемую структуру
         * записей. Для понимания этого нюанса обратитесь к файлу:
         *
         *     mimimi.modules/Sqlite/Sqlite.php  -->  метод __construct()
         *                           Sqlite.php  -->  метод autoCreate()
         *
         * Затем задаём в переменной $rows список строк, которые представляют
         * собой необходимые для приложения настройки, и наполняем ими только
         * что созданную таблицу базы данных.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function onInstall ( $db ) {
            $db->exec ( 'CREATE TABLE `settings` ( `name`               VARCHAR(32)  PRIMARY KEY,'
                                               . ' `value`              VARCHAR(1024),'
                                               . ' `editable_by_admin`  BOOLEAN      DEFAULT TRUE,'
                                               . ' `input_type`         VARCHAR(16)  DEFAULT "text",'
                                               . ' `input_caption`      VARCHAR(64),'
                                               . ' `input_attributes`   VARCHAR(128) )' );
            $rows = [
                [ 'last_scan',       '',             FALSE, '',       'Штамп времени последнего сканирования',         ''                    ],
                [ 'scan_pause',      300,            TRUE,  'number', 'Пауза между сканированиями в секундах',         'min="0" max="86400"' ],
                [ 'day_starts',      8,              TRUE,  'number', 'В каком часу просыпается сканер',               'min="0" max="23"'    ],
                [ 'day_ends',        20,             TRUE,  'number', 'В каком часу засыпает',                         'min="0" max="24"'    ],
                [ 'parser_active',   1,              TRUE,  'number', 'Сканер включен/нет',                            'min="0" max="1"'     ],
                [ 'tg_bot_token',    '',             TRUE,  'text',   'Токен бота, отправляющего вакансии в Телеграм', 'maxlength="80"'      ],
                [ 'max_bot_token',   '',             TRUE,  'text',   'Токен бота, отправляющего вакансии в MAX',      'maxlength="80"'      ],
                [ 'on_demo',         1,              TRUE,  'number', 'Демо режим включен/нет',                        'min="0" max="1"'     ],
                [ 'password_hash',   md5 ( 'demo' ), FALSE, '',       'Хеш пароля админа, кто будет редактором',       ''                    ],
                [ 'supervisor_hash', md5 ( 'test' ), FALSE, '',       'Хеш пароля супервизора',                        ''                    ]
            ];
            foreach ( $rows as $row ) {
                $db->exec ( 'INSERT INTO `settings` '
                          . 'VALUES ( ' . $db->makeSET ( $row, FALSE, FALSE ) . ' )' );
            }
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Получить значение настройки по её имени.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $name  Имя запрашиваемой настройки.
         * @return  string
         *
         * -----------------------------------------------------------------
         */

        public function get ( $name ) {
            return $this->owner->queryFirst ( 'SELECT `value` '
                                            . 'FROM `settings` '
                                            . 'WHERE `name` = ' . $this->owner->bind ( $name ) );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Сохранить значение настройки по её имени.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $name   Имя запрашиваемой настройки.
         * @param   mixed   $value  Новое значение настройки.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function set ( $name, $value ) {
            return $this->owner->exec ( 'UPDATE `settings` '
                                      . 'SET `value` = ' .  $this->owner->bind ( $value ) . ' '
                                      . 'WHERE `name` = ' . $this->owner->bind ( $name  ) );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Получить список настроек, позволенных для редактирования.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  array  Массив массивов, индексированных именами колонок извлечённых записей.
         *
         * -----------------------------------------------------------------
         */

        public function select ( ) {
            return $this->owner->queryRows ( 'SELECT * '
                                           . 'FROM `settings` '
                                           . 'WHERE `editable_by_admin` = 1' );
        }
    }
