<?php
/**
 * -------------------------------------------------------------------------
 *
 * Модуль субконтроллера Settings.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / Repost Vacancies
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    /**
     * ---------------------------------------------------------------------
     *
     * Подключаем из папки ядра фреймворка файл "mimimi.core/Module.php".
     * Там объявлен класс "MimimiModule", являющийся простейшей модульной
     * заготовкой. Этот класс подходит как основа для реализуемого ниже
     * модуля.
     *
     * ---------------------------------------------------------------------
     */

    mimimiInclude ( 'Module.php' );

    /**
     * ---------------------------------------------------------------------
     *
     * Создаём на основе класса той заготовки новый класс, в котором напишем
     * программный код текущего модуля. Обратите внимание как задано имя
     * нового класса - оно сложено из имени класса вышестоящего модуля, то
     * есть "MyMimimiControllersDashboard", и имени текущего PHP-файла без
     * расширения.
     *
     * ---------------------------------------------------------------------
     */

    class MyMimimiControllersDashboardSettings extends MimimiModule {

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Маршрутизировать запрос к сопоставленному макету шаблона.
         *
         * -----------------------------------------------------------------
         *
         * Этот метод обслуживает следующие URL-ы:
         *
         *     https://ваш.сайт/settings
         *
         * Генерация HTML-кода страницы пройдёт на основе такого макета:
         *
         *     repost.vacancies/Themes/default/dashboard/settings.tpl
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $url  (необязательный) Относительный URL запрошенной страницы.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function run ( $url = '' ) {
            if ( $this->owner->hasAdmin ( ) ) {
                $success = '';
                $rows    = $this->app->models->settings->select ( );
                $error   = $this->owner->hasForm ( ) ? $this->onUpdate ( $success, $rows )
                                                     : '';
                $data    = [ 'success'     => $success,
                             'error'       => $error,
                             'rows'        => $rows,
                             'button_name' => 'form' ];
                $this->owner->render ( 'dashboard/settings.tpl', $data );
                return;
            }
            // иначе это оказался не админ, значит перебрасываем его на страницу "Вход"
            $this->owner->gotoLogin ( );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Обработчик формы редактирования настроек.
         *
         * -----------------------------------------------------------------
         *
         * В случае успешного сохранения администратор будет принудительно
         * перенаправлен на главную страницу сайта. В противном случае он
         * останется на той же форме ввода, в которую будет возвращён текст
         * сообщения об ошибке или демо успехе.
         *
         * -----------------------------------------------------------------
         */

        protected function onUpdate ( & $success, & $previous ) {
            $changes = [ ];
            foreach ( $previous as & $row ) {
                $value = mimimiPost ( $row[ 'name' ], NULL );
                if ( is_null ( $value ) ) {
                    return 'Ошибка: Полученная форма настроек не соответствует ожидаемой!';
                }
                if ( $row[ 'value' ] != $value ) {
                    $value = $this->parse ( $row[ 'name' ], $row[ 'input_type' ], $value );
                    $row[ 'value' ] = $value;
                    $changes[ $row[ 'name' ] ] = $value;
                }
            }
            if ( $changes ) {
                if ( $this->owner->isDemo ( ) ) {
                    $success = 'Ок! Но так как Вы сейчас в демо режиме, сайт игнорирует сохранение любых изменений. Чтобы увидеть, как это работает в действительности, скачайте приложение и запустите у себя на сайте, отключив демо режим.';
                    return '';
                }
                $this->update ( $changes );
            }
            $this->owner->gotoHome ( );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Привести значение указанного поля формы ввода к ожидаемому типу.
         *
         * -----------------------------------------------------------------
         */

        // разбор значения из формы ввода
        protected function parse ( $name, $type, $value ) {
            switch ( $name ) {
                case 'scan_pause':    return       max ( 0, intval ( $value ) );
                case 'day_starts':    return min ( max ( 0, intval ( $value ) ), 23 );
                case 'day_ends':      return min ( max ( 0, intval ( $value ) ), 23 );
                case 'parser_active': return min ( max ( 0, intval ( $value ) ), 1 );
                case 'tg_bot_token':  return mb_substr ( $value, 0, 80, 'UTF-8' );
                case 'max_bot_token': return mb_substr ( $value, 0, 80, 'UTF-8' );
                case 'on_demo':       return min ( max ( 0, intval ( $value ) ), 1 );
                default:              return $type == 'number' ? intval ( $value )
                                                               : $value;
            }
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Запись отредактированных настроек в таблицу базы данных.
         *
         * -----------------------------------------------------------------
         */

        protected function update ( $rows ) {
            foreach ( $rows as $name => $value ) {
                $this->app->models->settings->set ( $name, $value );
            }
        }
    }
