<?php
/**
 * -------------------------------------------------------------------------
 *
 * Модуль контроллера "Админпанель".
 *
 * -------------------------------------------------------------------------
 *
 * Несмотря на то, что данный модуль служит всего лишь в качестве хранителя
 * в одной папке всех субконтроллеров административных страниц, здесь тем
 * не менее размещена часть методов, которые универсальны для нескольких
 * админстраниц сразу. Такими являются, как правило, методы обработки форм
 * ввода. Ведь подразумевается, что строение таких форм единое для всех
 * административных страниц сайта, а значит повторять эти обработки в каждом
 * субконтроллере не имеет смысла, проще вынести подобные методы на уровень
 * выше.
 *
 * Поскольку контроллер Dashboard фактически является owner-ом любого
 * дочернего субконтроллера, Вы можете оттуда обратиться к его методам вот
 * так:
 *
 *     $this->owner->желаемыйМетод();
 *     или
 *     $this->app->controllers->dashboard->желаемыйМетод();
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / Repost Vacancies
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    /**
     * ---------------------------------------------------------------------
     *
     * Подключаем из папки ядра фреймворка файл "mimimi.core/NodeModule.php".
     * Там объявлен класс "MimimiNodeModule", который является простейшей
     * заготовкой для модулей, поддерживающих дочерние, вызываемые через
     * оператор Стрелка. Этот класс подходит как основа для реализуемого
     * ниже модуля.
     *
     * ---------------------------------------------------------------------
     */

    mimimiInclude ( 'NodeModule.php' );

    /**
     * ---------------------------------------------------------------------
     *
     * Создаём на основе класса той заготовки новый класс, в котором напишем
     * программный код текущего модуля. Обратите внимание как задано имя
     * нового класса - оно сложено из имени класса вышестоящего модуля, то
     * есть "MyMimimiControllers", и имени текущего PHP-файла без расширения.
     *
     * ---------------------------------------------------------------------
     */

    class MyMimimiControllersDashboard extends MimimiNodeModule {

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Проверить что текущий пользователь уже авторизован.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  bool
         *
         * -----------------------------------------------------------------
         */

        public function hasAdmin ( ) {
            $value = $this->app->session->get ( 'user_marker' );
            return   $value != ''
                && ( $value == $this->app->models->settings->get ( 'password_hash'   )
                ||   $value == $this->app->models->settings->get ( 'supervisor_hash' ) );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Проверить что текущий пользователь является супервизором.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  bool
         *
         * -----------------------------------------------------------------
         */

        public function hasSupervisor ( ) {
            $value = $this->app->session->get ( 'user_marker' );
            return $value != ''
                && $value == $this->app->models->settings->get ( 'supervisor_hash' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Проверить что сайт находится в демо режиме.
         *
         * -----------------------------------------------------------------
         *
         * Обратите внимание, демо режим не действует для супервизора. Это
         * сделано для того, чтобы единственное доверенное лицо из комьюнити
         * могло менять что-то на демонстрационном сайте приложения без
         * необходимости вручную через нечто вроде phpMyAdmin сбрасывать
         * настройку "on_demo" в таблице "settings".
         *
         * Ведь будучи один раз включенной обычным администратором, его прав
         * будет уже недостаточно, чтобы вернуть эту настройку назад. Такое
         * возможно либо через ручное редактирование таблицы "settings", либо
         * с правами супервизора.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  bool
         *
         * -----------------------------------------------------------------
         */

        public function isDemo ( ) {
            return $this->app->models->settings->get ( 'on_demo' ) == TRUE
                && $this->hasSupervisor ( ) == FALSE;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Проверить что пользователем только что прислана форма ввода.
         *
         * -----------------------------------------------------------------
         *
         * Подразумевается, что любая форма на сайте будет иметь кнопку
         * отправки по имени "form" как простейший механизм подтверждения,
         * что пользователь в самом деле заполнил форму ввода и отправил.
         * Для понимания этого нюанса взгляните как сделаны атрибуты "name"
         * и "value" у кнопки <button> в следующих макетах:
         *
         *     repost.vacancies/Themes/default/dashboard/login.tpl
         *     repost.vacancies/Themes/default/dashboard/logout.tpl
         *     repost.vacancies/Themes/default/dashboard/password-change.tpl
         *     repost.vacancies/Themes/default/dashboard/settings.tpl
         *     repost.vacancies/Themes/default/dashboard/tasks-add.tpl
         *     repost.vacancies/Themes/default/dashboard/tasks-copy.tpl
         *     repost.vacancies/Themes/default/dashboard/tasks-delete.tpl
         *     repost.vacancies/Themes/default/dashboard/tasks-edit.tpl
         *     repost.vacancies/Themes/default/dashboard/vacancies-delete.tpl
         *     repost.vacancies/Themes/default/dashboard/vacancies-edit.tpl
         *
         * Заметьте также, в каких контроллерах готовится для макетов некий
         * параметр "button_name", который как раз и содержит упомянутое имя
         * для главной кнопки формы ввода:
         *
         *     repost.vacancies/Controllers/Dashboard/Login/Login.php
         *     repost.vacancies/Controllers/Dashboard/Settings/Settings.php
         *     repost.vacancies/Controllers/Dashboard/Tasks/Tasks.php
         *     repost.vacancies/Controllers/Dashboard/Vacancies/Vacancies.php
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  bool
         *
         * -----------------------------------------------------------------
         */

        public function hasForm ( ) {
            return mimimiPost ( 'form' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Проверить что согласие пользователя получено.
         *
         * -----------------------------------------------------------------
         *
         * Подразумевается, что любая форма на сайте, требующая от человека
         * подтвердить некое намерение, будет иметь флажок по имени "agree".
         * Для понимания этого нюанса взгляните как сделаны атрибуты "name"
         * и "value" у флажка <input type="checkbox"> в следующих макетах:
         *
         *     repost.vacancies/Themes/default/dashboard/logout.tpl
         *     repost.vacancies/Themes/default/dashboard/tasks-delete.tpl
         *     repost.vacancies/Themes/default/dashboard/vacancies-delete.tpl
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  bool
         *
         * -----------------------------------------------------------------
         */

        public function hasAgreed ( ) {
            return $this->hasForm ( )
                && mimimiPost ( 'agree' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Перебросить пользователя на другой URL.
         *
         * -----------------------------------------------------------------
         *
         * Обратите внимание, выполняется редирект "307 Temporary Redirect",
         * чтобы браузер не кешировал адрес перехода, а поисковик не склеивал
         * исходный и конечный адрес.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $url  Относительный URL желаемой страницы.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function redirect ( $url ) {
            $site = mimimiSite ( FALSE ) . mimimiRoot ( FALSE );
            mimimiStop ( $site . $url, 307 );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Перебросить пользователя на главную страницу.
         *
         * -----------------------------------------------------------------
         *
         * Обратите внимание, выполняется редирект "307 Temporary Redirect",
         * чтобы браузер не кешировал адрес перехода, а поисковик не склеивал
         * исходный и конечный адрес.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function gotoHome ( ) {
            $this->redirect ( '' );
        }

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Перебросить пользователя на страницу авторизации.
         *
         * -----------------------------------------------------------------
         *
         * Обратите внимание, выполняется редирект "307 Temporary Redirect",
         * чтобы браузер не кешировал адрес перехода, а поисковик не склеивал
         * исходный и конечный адрес.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function gotoLogin ( ) {
            $this->redirect ( 'login' );
        }

        // генерация контента страницы

        /**
         * -----------------------------------------------------------------
         *
         * Метод: Сгенерировать контент админской страницы.
         *
         * -----------------------------------------------------------------
         *
         * Генерация страниц админпанели отличается от обычных всего лишь
         * серверными заголовками, которые полностью запрещают кеширование
         * и индексирование подобных страниц.
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $file  Путь к файлу макета, на основе которого желаете генерировать страницу.
         *                         Указывается относительно папки шаблона, то есть относительно "repost.vacancies/Themes/default/".
         * @param   mixed   $data  (необязательный) Данные для передачи в макет, если там необходимы.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        public function render ( $file, $data = FALSE ) {
            $this->app->views->html->administrative ([ $file, $data ]);
        }

        /**
         * -----------------------------------------------------------------
         *
         * Инициализируем симулятор "неймспейса", так как в силу конструкции
         * класса "MimimiNodeModule" такое действие предписано совершать в
         * каждом наследуемом классе, если начиная с его узла тоже требуется
         * поддержать способность обращаться к его дочерним модулям через
         * оператор Стрелка (->).
         *
         * Как Вы можете видеть, в папке этого модуля находятся несколько
         * субконтроллеров: Login, Settings, Tasks, Vacancies, к которым
         * нам придётся обращаться в любом случае. Поэтому мы и выполнили
         * сейчас инициализацию "неймспейса".
         *
         * Под неймспейсом понимается уникализация имён вложенных модульных
         * классов. Посмотрите как назван относительно текущего класса любой
         * класс дочернего модуля.
         *
         * -----------------------------------------------------------------
         *
         * @var    string
         * @access protected
         *
         * -----------------------------------------------------------------
         */

        protected $myNodeFile = __FILE__;
    }
