<?php
/**
 * -------------------------------------------------------------------------
 *
 * The database module for selecting user entries.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / IDE skeleton
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    mimimiInclude ( 'ModuleWithTable.php' );
    class MyMimimiUsers extends MimimiModuleWithTable {

        /**
         * -----------------------------------------------------------------
         *
         * Specify a name of the database table to store users.
         *
         * -----------------------------------------------------------------
         *
         * @access public
         * @var    string
         *
         * -----------------------------------------------------------------
         */

        public $table = 'users';

        /**
         * -----------------------------------------------------------------
         *
         * Define a database table structure.
         *
         * -----------------------------------------------------------------
         *
         * @access protected
         * @var    array
         *
         * -----------------------------------------------------------------
         */

        protected $tableFields = [
                      '`id`     BIGINT(20)   NOT NULL  AUTO_INCREMENT  COMMENT "user system identifier"',
                      '`login`  VARCHAR(50)  NOT NULL                  COMMENT "login"',
                      '`hash`   VARCHAR(50)  NOT NULL                  COMMENT "password hash"'
                  ];

        /**
         * -----------------------------------------------------------------
         *
         * Define a list of table keys to speed up the database operations
         * related to users.
         *
         * -----------------------------------------------------------------
         *
         * @access protected
         * @var    array
         *
         * -----------------------------------------------------------------
         */

        protected $tableKeys = [
                      'PRIMARY KEY ( `id`    )',
                      'UNIQUE KEY  ( `login` )',
                      'KEY         ( `hash`  )'
                  ];

        /**
         * -----------------------------------------------------------------
         *
         * Returns the user identifier given their login.
         *
         * -----------------------------------------------------------------
         *
         * Please note that "t1" below is an alias for the database table
         * "users". Note also that the filter used below will actually
         * result in the following MySQL query:
         *
         *     SELECT t1.id
         *     FROM   users AS t1
         *     WHERE  t1.login = $login AND
         *            t1.hash  = $hash
         *     LIMIT  1
         *
         * -----------------------------------------------------------------
         *
         * @public
         * @param   string  $login  The user login you are looking for.
         * @param   string  $pass   (optional) The user password if you need to authorize it.
         * @return  int
         *
         * -----------------------------------------------------------------
         */

        public function getId ( $login, $pass = null ) {
            $filter = [
                'select' => [ 't1.id'    => TRUE ],
                /* where */   't1.login' => $login
            ];
            if ( is_string ( $pass ) ) {
                $hash = md5 ( $pass );
                $filter[ 't1.hash' ] = $hash;
            }
            $entry = $this->select ( $filter );
            return isset ( $entry[ 'id' ] )
                         ? $entry[ 'id' ]
                         : 0;
        }

        /**
         * -----------------------------------------------------------------
         *
         * Renames an entry column and/or modifies its value.
         *
         * -----------------------------------------------------------------
         *
         * @protected
         * @param   string  $name   The name of entry column that may be renamed.
         * @param   mixed   $value  The value of the column that may be modified.
         * @return  void
         *
         * -----------------------------------------------------------------
         */

        protected function renameField ( & $name, & $value ) {
            switch ( $name ) {
                case 'password':
                     $name  = 'hash';
                     $value = md5 ( $value );
                     break;
            }
        }

        /**
         * -----------------------------------------------------------------
         *
         * Specify demo rows that will be used as default user entries if
         * the database does not have a table named "users". In this case,
         * all demo rows will be automatically added to the newly created
         * table.
         *
         * -----------------------------------------------------------------
         *
         * @access protected
         * @var    array
         *
         * -----------------------------------------------------------------
         */

        protected $demoRows = [
            [
                'id'       => 1,
                'login'    => 'demo',
                'password' => 'demo'    // <-- it will be converted to a hash (see method renameField() above for details)
            ]
        ];
    };
