/**
 * -------------------------------------------------------------------------
 *
 * A JavaScript module to configure the built-in HTML editor.
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Examples / Static Pages only
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 *
 * -------------------------------------------------------------------------
 */

    /**
     * ---------------------------------------------------------------------
     *
     * First, you should load the free plugins you want to use in the
     * built-in HTML editor.
     *
     * ---------------------------------------------------------------------
     */

    import { ClassicEditor,
             CloudServices,
             Essentials,
             Font,
             GeneralHtmlSupport,
             Heading,
                 HeadingButtonsUI,
             Paragraph,
                 ParagraphButtonUI,
                     Bold,
                     Italic,
                     Strikethrough,
                     Underline,
                     Code,
                     CodeBlock,
                     Subscript,
                     Superscript,
                     HorizontalLine,
             Alignment,
                 AlignmentEditing,
                 AlignmentUI,
             Link,
             Image,
                 ImageUpload,
                     Base64UploadAdapter,
                     AutoImage,
                 ImageInsert,
                 ImageToolbar,
                     ImageCaption,
                     ImageTextAlternative,
                     ImageStyle,
                     ImageInline,
                     ImageResize,
                         ImageResizeButtons,
                             ImageResizeEditing,
                             ImageResizeHandles,
                     LinkImage,
             MediaEmbed,
             Table,
                 TableToolbar,
             BlockQuote,
             List,
                 ListProperties,
                 Indent,
                     IndentBlock,
                 TodoList,
             PasteFromOffice,
             ShowBlocks,
             RemoveFormat,
             SourceEditing,
             WordCount,

             /**
              * ------------------------------------------------------------
              *
              * And also a few components to develop a plugin "bbCodes" below.
              *
              * ------------------------------------------------------------
              */

             Plugin,
             DropdownButtonView,
             BalloonPanelView,
             ButtonView } from 'ckeditor5';

    /**
     * ---------------------------------------------------------------------
     *
     * And you should also load the paid plugins you want to use. To do this,
     * uncomment the following line below and specify the names of the
     * desired paid plugins there.
     *
     * ---------------------------------------------------------------------
     */

    /* import { ImportWord     ,
                MultiLevelList ,
                Template       } from 'ckeditor5-premium-features'; */

    /**
     * ---------------------------------------------------------------------
     *
     * Now we need to develop the plugin we discussed above. Note that this
     * plugin uses the STATICPAGESONLY_CKEDITOR_BBCODES constant defined
     * earlier (to understand this moment, please see the snippet file
     * "static.pages.only/Themes/default/snippets/editor-scripts.tpl").
     *
     * ---------------------------------------------------------------------
     */

    class bbCodes extends Plugin {
        init ( ) {
            const editor = this.editor;
            editor.ui.componentFactory.add ( 'bbcodes',
                ( ) => {
                    let i, button;
                    const dropdown = new DropdownButtonView ( ),
                          panel    = new BalloonPanelView ( ),
                          callback = ( eventInfo ) => {
                                         editor.model.change (
                                             ( writer ) => {
                                                 editor.model.insertContent (
                                                     writer.createText ( '[insert=' + eventInfo.source.labelUrl + ']' )
                                                 );
                                             }
                                         );
                                     };
                    dropdown.set ( { label:    'Insert dynamic block' ,
                                     tooltip:  true              ,
                                     withText: false             } );
                    for ( i = 0; i < STATICPAGESONLY_CKEDITOR_BBCODES.length; i++ ) {
                        button = new ButtonView ( );
                        button.set ( { label:    STATICPAGESONLY_CKEDITOR_BBCODES[ i ].replace ( /\s*\|.*$/, '' ) ,
                                       labelUrl: STATICPAGESONLY_CKEDITOR_BBCODES[ i ].replace ( /^.*\|\s*/, '' ) ,
                                       withText: true                                  } );
                        button.on ( 'execute', callback );
                        panel.content.add ( button );
                    }
                    button = new ButtonView ( );
                    button.set ( { label:    'edit this list...' ,
                                   withText: true               } );
                    button.on ( 'execute',
                        ( eventInfo ) => {
                            window.location.replace ( 'bbcodes' );
                        }
                    );
                    panel.content.add ( button );
                    dropdown.children.add ( panel );
                    dropdown.on ( 'execute',
                        ( eventInfo ) => {
                            panel.isVisible = ! panel.isVisible;
                        }
                    );
                    return dropdown;
                }
            );
        }
    }

    /**
     * ---------------------------------------------------------------------
     *
     * Next, you should define the parameters for how the tools will be
     * displayed in the built-in HTML editor.
     *
     * ---------------------------------------------------------------------
     */

    const node = document.querySelector ( 'body > .wrapper > .content > .form > .left' );
    if ( node ) {

        const params = {
            plugins: [
                CloudServices,
                Essentials,
                Font,
                GeneralHtmlSupport,
                Heading,
                    HeadingButtonsUI,
                Paragraph,
                    ParagraphButtonUI,
                        Bold,
                        Italic,
                        Strikethrough,
                        Underline,
                        Code,
                        CodeBlock,
                        Subscript,
                        Superscript,
                        HorizontalLine,
                Alignment,
                    AlignmentEditing,
                    AlignmentUI,
                Link,
                Image,
                    ImageUpload,
                        Base64UploadAdapter,
                        AutoImage,
                    ImageInsert,
                    ImageToolbar,
                        ImageCaption,
                        ImageTextAlternative,
                        ImageInline,
                        ImageStyle,
                        ImageResize,
                            ImageResizeButtons,
                                ImageResizeEditing,
                                ImageResizeHandles,
                        LinkImage,
                MediaEmbed,
                Table,
                    TableToolbar,
                BlockQuote,
                List,
                    ListProperties,
                    Indent,
                        IndentBlock,
                    TodoList,
                PasteFromOffice,
                ShowBlocks,
                RemoveFormat,
                SourceEditing,
                WordCount,

                /**
                 * ---------------------------------------------------------
                 *
                 * Note that you must specify the names of all paid plugins
                 * whose buttons you want to see on the toolbar in the
                 * following block.
                 *
                 * ---------------------------------------------------------
                 *
                 */

                ...( STATICPAGESONLY_CKEDITOR_KEY !== 'GPL' ? [ ImportWord     ,
                                                                MultiLevelList ,
                                                                Template       ] : [ ] ),

                /**
                 * ---------------------------------------------------------
                 *
                 * And also specify the developed plugin "bbCodes".
                 *
                 * ---------------------------------------------------------
                 *
                 */

                bbCodes
            ],

            toolbar: {
                items: [
                    'undo',
                    'redo',
                    '|',
                    'heading',
                        // 'paragraph',
                    '|',
                    'importWord',    // it is a paid plugin
                    'selectAll',
                    '|',
                    'bold',
                    'italic',
                    'underline',
                    'strikethrough',
                    '|',
                    'code',
                    'codeBlock',
                    'subscript',
                    'superscript',
                    'horizontalLine',
                    '|',
                    // 'alignment',
                        'alignment:left',
                        'alignment:center',
                        'alignment:right',
                        'alignment:justify',
                    '|',
                    'link',
                    // 'insertImage',
                        'insertImageViaUrl',
                        'uploadImage',
                        // 'resizeImage',
                    'insertTable',
                    'mediaEmbed',
                    'blockQuote',
                    'bbcodes',
                    'insertTemplate',    // it is a paid plugin
                    '|',
                    'bulletedList',
                    'numberedList',
                    'multilevelList',    // it is a paid plugin
                        'indent',
                        'outdent',
                    // '|',
                    // 'todoList',
                    '|',
                    'showBlocks',
                    'removeFormat',
                    'sourceEditing'
                ]
            },

            /**
             * -------------------------------------------------------------
             *
             * Configure the "heading" tool.
             *
             * -------------------------------------------------------------
             */

            heading: {
                options: [
                    { model: 'paragraph',             title: 'Paragraph', class: 'ck-heading_paragraph', icon: '', upcastAlso: [] },
                    { model: 'heading1',  view: 'h1', title: 'Heading 1', class: 'ck-heading_heading1',  icon: '', upcastAlso: [] },
                    { model: 'heading2',  view: 'h2', title: 'Heading 2', class: 'ck-heading_heading2',  icon: '', upcastAlso: [] },
                    { model: 'heading3',  view: 'h3', title: 'Heading 3', class: 'ck-heading_heading3',  icon: '', upcastAlso: [] },
                    { model: 'heading4',  view: 'h4', title: 'Heading 4', class: 'ck-heading_heading4',  icon: '', upcastAlso: [] },
                    { model: 'heading5',  view: 'h5', title: 'Heading 5', class: 'ck-heading_heading5',  icon: '', upcastAlso: [] },
                    { model: 'heading6',  view: 'h6', title: 'Heading 6', class: 'ck-heading_heading6',  icon: '', upcastAlso: [] }
                ]
            },

            /**
             * -------------------------------------------------------------
             *
             * Configure the "alignment" tool.
             *
             * -------------------------------------------------------------
             */

            aligment: {
                options: [
                    { name: 'left',    className: ''        },
                    { name: 'center',  className: 'center'  },
                    { name: 'right',   className: 'right'   },
                    { name: 'justify', className: 'justify' }
                ]
            },

            /**
             * -------------------------------------------------------------
             *
             * Configure the "insertImage" tool.
             *
             * -------------------------------------------------------------
             */

            image: {
                insert: {
                    integrations: [
                        'upload',
                        'url'
                    ]
                },
                toolbar: [
                    'toggleImageCaption',
                    'imageTextAlternative',
                    '|',
                    'imageStyle:block',
                    'imageStyle:inline',
                    'imageStyle:wrapText',
                    '|',
                    'resizeImage:50%',
                    'resizeImage:100%',
                    'resizeImage:original',
                    '|',
                    'linkImage'
                ],
                resizeOptions: [
                    { name:  'resizeImage:50%' ,
                      value: '50'              ,
                      icon:  'medium'          },
                    { name:  'resizeImage:100%' ,
                      value: '100'              ,
                      icon:  'large'            },
                    { name:  'resizeImage:original' ,
                      value: null                   ,
                      icon:  'original'             }
                ],
                resizeUnit: '%'
            },

            /**
             * -------------------------------------------------------------
             *
             * Configure the "insertTable" tool.
             *
             * -------------------------------------------------------------
             */

            table: {
                contentToolbar: [
                    'tableColumn',
                    'tableRow',
                    'mergeTableCells'
                ]
            },

            /**
             * -------------------------------------------------------------
             *
             * Configure the "GeneralHtmlSupport" plugin.
             *
             * For more details, see this page
             * https://ckeditor.com/docs/ckeditor5/latest/features/html/general-html-support.html
             *
             * -------------------------------------------------------------
             */

            htmlSupport: {
                disallow: [ // Disable all HTML features.
                            { name:       /.*/  ,
                              attributes: true  ,
                              classes:    true  ,
                              styles:     true  } ],
                allow:    [ // Enable some HTML features.
                            { name: 'main'                                                                                                               },
                            { name: 'article'                                                                                                            },
                            { name:     'aside'              , classes: true                                                                             },
                            { name:     'section'            , classes: true                                                                             },
                            { name:         'div'            , classes: true                                                                             },
                            { name:         'h1'             , classes: [ 'to-left', 'center', 'to-right', 'justify' ]                                   },
                            { name:         'h2'             , classes: [ 'to-left', 'center', 'to-right', 'justify' ]                                   },
                            { name:         'h3'             , classes: [ 'to-left', 'center', 'to-right', 'justify' ]                                   },
                            { name:         'h4'             , classes: [ 'to-left', 'center', 'to-right', 'justify' ]                                   },
                            { name:         'h5'             , classes: [ 'to-left', 'center', 'to-right', 'justify' ]                                   },
                            { name:         'h6'             , classes: [ 'to-left', 'center', 'to-right', 'justify' ]                                   },
                            { name:         'blockquote'                                                                                                 },
                            { name:             'p'          , classes: [ 'to-left', 'center', 'to-right', 'justify' ]                                   },
                            { name:                 'b'                                                                                                  },
                            { name:                 'strong'                                                                                             },
                            { name:                 'i'                                                                                                  },
                            { name:                 'em'                                                                                                 },
                            { name:                 's'                                                                                                  },
                            { name:                 'u'                                                                                                  },
                            { name:                 'sub'                                                                                                },
                            { name:                 'sup'                                                                                                },
                            { name:                 'a'      , attributes: { href: true, title: true, target: true, rel: true }                          },
                            { name:         'ul'                                                                                                         },
                            { name:         'ol'                                                                                                         },
                            { name:             'li'                                                                                                     },
                            { name:         'figure'         , classes: [ 'image', 'table', 'to-left', 'center', 'to-right', 'justify' ]                 },
                            { name:             'img'        , attributes: { src: true, alt: true, rel: true, width: true, height: true, loading: true } },
                            { name:             'figcaption'                                                                                             },
                            { name:         'table'          , attributes: { border: true, cellpadding: true, cellspacing: true, width: true }           },
                            { name:             'thead'                                                                                                  },
                            { name:             'tbody'                                                                                                  },
                            { name:                 'tr'                                                                                                 },
                            { name:                     'th' , attributes: { nowrap: true, colspan: true, rowspan: true, width: true }                   },
                            { name:                     'td' , attributes: { nowrap: true, colspan: true, rowspan: true, width: true }                   },
                            { name:             'tfoot'                                                                                                  },
                            { name:         'pre'                                                                                                        },
                            { name:             'code'                                                                                                   },
                            { name:         'hr'                                                                                                         } ]
            },

            /**
             * -------------------------------------------------------------
             *
             * Configure other elements.
             *
             * -------------------------------------------------------------
             */

            placeholder: 'Type or paste your content here!'
        };

        /**
         * -----------------------------------------------------------------
         *
         * And finally, we create an instance of the HTML editor. After
         * this, we will be able to access the editor object through the
         * public property "window.editor".
         *
         * -----------------------------------------------------------------
         */

        params.licenseKey = STATICPAGESONLY_CKEDITOR_KEY;
        ClassicEditor.create ( node, params                          )
                     .then   ( editor => {
                                   window.editor = editor;
                                   const widget = editor.plugins.get( 'WordCount' ).wordCountContainer;
                                   document.querySelector ( '.ck.ck-editor__main' ).appendChild ( widget );
                               } )
                     .catch  ( error  => { console.error ( error ) } );
    }
