<?php
/**
 * -------------------------------------------------------------------------
 *
 * Let's imagine that you are now in the initial stage and are trying to
 * develop a website containing the following set of features: a home page
 * with a list of posted newspapers, the newspaper pages themselves, three
 * backend pages (add, edit, delete) for your team of editors, an admin
 * page for editing website settings, two SEO pages (sitemap, robots.txt),
 * and a Error 404 page.
 *
 * To program a website that matches the one described above, you need to go
 * through six stages: implement an application module to receive browser
 * requests, program a new one or just use a basic module to parse segments
 * of the requested URL, program a new one or just use a basic settings
 * module to work with its database entries, program a newspaper module to
 * work with its entries, mark up website theme templates for each planned
 * page type, and program a new or extend a basic helper module containing
 * some routines for use in page templates.
 *
 * Let's start right here. And as development progresses, we will briefly
 * comment on each stage and its steps.
 *
 * -------------------------------------------------------------------------
 *
 * STAGE 1: It's about creating an application module. This stage will
 *          consist of three steps: declaring the required class, writing
 *          a routing method, writing an identification method for your
 *          editors and administrators.
 *
 * -------------------------------------------------------------------------
 *
 * STEP 1:  You should connect the base app module using the "mimimiInclude"
 *          directive. Then you should declare a class "MyMimimiApplication"
 *          which extends the base app class. Next, you will need to follow
 *          STEP 2 and STEP 3 detailed below to program this application
 *          module.
 *
 * -------------------------------------------------------------------------
 *
 * @license  GPL-2.0  https://opensource.org/license/gpl-2-0/
 * @author   MiMiMi Community
 *
 * -------------------------------------------------------------------------
 */

mimimiLoad    ( 'RoutinesWeb.php' );
mimimiInclude ( 'Application.php' );

class MyMimimiApplication extends MimimiApplication {

    /**
     * ---------------------------------------------------------------------
     *
     * STEP 2: You should declare and implement a public method to parse and
     *         route the incoming URL to the appropriate template file for
     *         your website theme. Let's use the "run" method for this
     *         purpose, which is usually used to render the requested web
     *         page. We'll get the page URL using the "mimimiUri" directive
     *         and then match it with predefined URLs that are assigned
     *         a specific template file and also read the necessary data for
     *         that template. The predefined URLs are as follows:
     *             https://your.website/                       - to display the homepage
     *             https://your.website/robots.txt             - to provide scanning directives for crawlers
     *             https://your.website/sitemap                - to provide an XML sitemap for crawlers
     *             https://your.website/settings               - to edit your website settings
     *             https://your.website/add                    - to add a new newspaper
     *             https://your.website/A-NEWSPAPER-URL/delete - to delete an existing newspaper
     *             https://your.website/A-NEWSPAPER-URL/edit   - to edit an existing newspaper
     *             https://your.website/A-NEWSPAPER-URL        - to display the newspaper
     *
     * ---------------------------------------------------------------------
     *
     * @public
     * @param   mixed  $params  (optional) Some parameters from the calling process.
     * @return  void
     *
     * ---------------------------------------------------------------------
     */

    public function run ( $params = '' ) {
        $data     = FALSE;
        $template = '404.tpl';

        // start routing by URL
        $url = mimimiUri( FALSE );
        switch ( $url ) {

            // if the URL points to the home page, get
            // a list of newspapers (ignoring their
            // statuses if an admin is currently logged in)
            case '':
                $evenModerated = $this->isAdmin( FALSE );
                $data          = $this->newspapers->getItems( $evenModerated );
                $template      = 'home.tpl';
                break;

            // if the URL points to an SEO page
            case 'robots.txt':
                $template = 'robots.tpl';
                break;
            case 'sitemap':
                $data     = $this->newspapers->getItems();
                $template = 'newspapers-sitemap.tpl';
                break;

            // if the URL points to the "Edit Settings" page
            case 'settings':
                if ( $this->isAdmin( FALSE ) ) {
                    $data     = $this->settings->getItems();
                    $template = 'settings-edit.tpl';
                }
                break;

            // if the URL points to the "Add Newspaper" page
            case 'add':
                if ( $this->isAdmin() ) {
                    $template = 'newspapers-add.tpl';
                }
                break;

            // otherwise, if the command (i.e. the last URL segment)
            //            points to the "Delete Newspaper" page
            //            or the "Edit Newspaper" page, get the
            //            requested newspaper (ignoring its status
            //            if an admin is currently logged in)
            default:
                $command = $this->url->parseCommand( $url );
                switch ( $command ) {
                    case 'delete':
                    case 'edit':
                        if ( $url != $command ) {
                            if ( $this->isAdmin() ) {
                                $url = $this->url->removeCommand( $url );
                                $evenModerated = $this->isAdmin( FALSE );
                                $data          = $this->newspapers->getItem( $url, $evenModerated );
                                if ( $data ) {
                                    $template = 'newspapers-' . $command . '.tpl';
                                }
                            }
                        }
                        break;

                    // otherwise the URL points to the newspaper page,
                    //           and then we get the requested newspaper
                    //           (ignoring its status if an admin is
                    //           currently logged in)
                    default:
                        $evenModerated = $this->isAdmin( FALSE );
                        $data          = $this->newspapers->getItem( $url, $evenModerated );
                        if ( $data ) {
                            $template = 'newspapers-page.tpl';
                        }
                }
        }

        // render HTML page using routed template and retrieved data
        $this->helper->run();
        mimimiModule( [ $template, $data ] );
    }

    /**
     * ---------------------------------------------------------------------
     *
     * STEP 3: Finally, you should declare and implement a public method to
     *         check if the current user has administrator or editor role.
     *         Verification is performed using a list of allowed IP addresses
     *         specified in the corresponding setting. Please note that this
     *         simple authorization scheme allows you to get rid of a feature
     *         like "Sign In" on your website, but it can be inconvenient
     *         when accessing the website through dynamic IP addresses or
     *         a public hotspot.
     *
     * ---------------------------------------------------------------------
     *
     * @public
     * @param   bool  $any  (optional) TRUE  if you want to check membership of any role.
     *                                 FALSE if you want to check only the admin role.
     * @return  bool        TRUE if the current user is allowed or website is in demo mode.
     *
     * ---------------------------------------------------------------------
     */

    public function isAdmin ( $any = TRUE ) {
        $on = $this->settings->getValue( 'on_production' );
        if ( $on ) {
            $me = mimimiServer( 'REMOTE_ADDR' );
            $roles = $any ? [ 'ips_editor', 'ips_admin' ]
                          : [ 'ips_admin'               ];
            foreach ( $roles as $param ) {
                $list = $this->settings->getValue( $param );
                if ( in_array( $me, $list ) ) {
                    return TRUE;
                }
            }
            return FALSE;
        }
        return TRUE;
    }
};

/**
 * -------------------------------------------------------------------------
 *
 * That's all right! You should now examine the "newspaper/Url/Url.php" file
 * that implements STAGE 2.
 *
 * -------------------------------------------------------------------------
 */