<?php
/**
 * -------------------------------------------------------------------------
 * Miniature library of helper routines for system purposes.
 *
 * -------------------------------------------------------------------------
 *
 * Implemented routines below are:
 *     mimimiDump
 *     mimimiModule
 *     mimimiCookie
 *     mimimiFiles
 *     mimimiGet
 *     mimimiPost
 *     mimimiRequest
 *     mimimiSession
 *     mimimiSite
 *     mimimiRoot
 *     mimimiUri
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Core
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 * @license    GPL-2.0
 *             https://opensource.org/license/gpl-2-0/
 * -------------------------------------------------------------------------
 */

    /* =========================================================
    |                                                          |
    |   Define a global routine to output a required THEME'S   |
    |   TEXTUAL file                                           |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $filename = STRING   = relative path and name,     |
    |                              for example, css/style.css  |
    |       $fromCore = BOOLEAN  = TRUE  if look in CORE folder|
    |                            = FALSE if look in APP folder |
    |                            = NULL if look in both folders|
    |       $toDisplay = BOOLEAN = TRUE if echo it to browser  |
    |                              FALSE if returns as result  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       STRING = empty string if $toDisplay is TRUE,       |
    |                file content if $toDisplay is FALSE       |
    |                                                          |
    ========================================================= */

    if ( ! function_exists('mimimiDump') ) {
        function mimimiDump ( $filename, $fromCore = FALSE, $toDisplay = TRUE ) {
            $path = mimimiBasePath ( );
            $theme = 'Themes/' .
                     (
                         $fromCore
                         ? MIMIMI_CORE_THEME
                         : MIMIMI_APP_THEME
                     ) .
                     '/';

            /* ---------------------------------
            |                                  |
            |   Build a plan for dumping       |
            |                                  |
            --------------------------------- */

            $plan = [
                MIMIMI_INSTALL_FOLDER => $fromCore !== TRUE,
                MIMIMI_APP_FOLDER     => $fromCore !== TRUE,
                MIMIMI_MODULES_FOLDER => $fromCore !== FALSE,
                MIMIMI_CORE_FOLDER    => $fromCore !== FALSE
            ];

            /* ---------------------------------
            |                                  |
            |   Loop through plan's items      |
            |                                  |
            --------------------------------- */

            $result = '';
            foreach ($plan as $folder => $enabled) {
                if ($enabled) {
                    $file = $path . $folder . $theme . $filename;
                    if (file_exists($file)) {

                        /* -------------------------
                        |                          |
                        |   Try to dump the file   |
                        |                          |
                        ------------------------- */

                        if ($toDisplay) {
                            readfile($file);
                        } else {
                            $result = file_get_contents($file);
                        }
                        break;
                    }
                }
            }
            return $result;
        }
    }

    /**
     * ---------------------------------------------------------------------
     * Global routine to load a required THEME'S TEMPLATE file.
     *
     * Note: This routine should be considered as syntactic sugar to be used
     *       only in the theme files of the framework's modified version.
     *
     * @public
     * @param   string|array  $filename  STRING Relative path and name, for example, blocks/header.php.
     *                                   ARRAY  The first item is a relative path and name, other items are the input parameters.
     * @param   bool          $fromCore  True if look in the CORE folder,
     *                                   False if look in the APP folder.
     * @return  void
     * ---------------------------------------------------------------------
     */

    if ( ! function_exists('mimimiModule') ) {
        function mimimiModule ( $filename, $fromCore = FALSE ) {
            $params = FALSE;
            if ( is_array($filename) ) {
                $params = $filename;
                $filename = array_shift($params);
                if ( is_array($params) ) {
                    if ( count($params) == 1 ) {
                        $params = reset($params);
                    }
                }
            }
            if ( is_string($filename) ) {
                $theme = 'Themes/' .
                         (
                             $fromCore
                             ? MIMIMI_CORE_THEME
                             : MIMIMI_APP_THEME
                         ) .
                         '/';
                mimimiInclude(
                    $theme . $filename,
                    $fromCore == TRUE,
                    FALSE,
                    $params
                );
            }
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named COOKIE's   |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    if ( ! function_exists('mimimiCookie') ) {
        function mimimiCookie ( $param, $def = NULL ) {
            return isset($_COOKIE[$param])
                   ? $_COOKIE[$param]
                   : $def;
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named FILES's    |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    if ( ! function_exists('mimimiFiles') ) {
        function mimimiFiles ( $param, $def = NULL ) {
            return isset($_FILES[$param])
                   ? $_FILES[$param]
                   : $def;
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named GET's      |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    if ( ! function_exists('mimimiGet') ) {
        function mimimiGet ( $param, $def = '' ) {
            return isset($_GET[$param])
                   ? $_GET[$param]
                   : $def;
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named POST's     |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    if ( ! function_exists('mimimiPost') ) {
        function mimimiPost ( $param, $def = '' ) {
            return isset($_POST[$param])
                   ? $_POST[$param]
                   : $def;
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named REQUEST's  |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    if ( ! function_exists('mimimiRequest') ) {
        function mimimiRequest ( $param, $def = '' ) {
            return isset($_REQUEST[$param])
                   ? $_REQUEST[$param]
                   : $def;
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to retrieve a named SESSION's  |
    |   parameter                                              |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $param = STRING = name of the parameter            |
    |       $def   = MIXED  = a default value                  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       MIXED = retrieved parameter                        |
    |                                                          |
    ========================================================= */

    if ( ! function_exists('mimimiSession') ) {
        function mimimiSession ( $param, $def = NULL ) {
            return isset($_SESSION[$param])
                   ? $_SESSION[$param]
                   : $def;
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to display or retrieve the     |
    |   website domain's absolute URI                          |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $toDisplay = BOOLEAN = TRUE if echo it to browser  |
    |                              FALSE if returns as result  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       STRING = retrieved URI (with trailing slash)       |
    |                                                          |
    ========================================================= */

    if ( ! function_exists('mimimiSite') ) {
        function mimimiSite ( $toDisplay = TRUE ) {
            $domain = mimimiServer('HTTP_HOST');
            $scheme = mimimiServer('HTTPS')
                      ? 'https'
                      : mimimiServer('REQUEST_SCHEME');
            if ($scheme != '') {
                $scheme .= ':';
            }
            $value = $scheme . '//' . $domain . '/';
            if ($toDisplay) {
                echo $value;
            } else {
                return $value;
            }
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to display or retrieve the     |
    |   relative path's offset to the Core's folder            |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $toDisplay = BOOLEAN = TRUE if echo it to browser  |
    |                              FALSE if returns as result  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       STRING = retrieved path (without leading slash)    |
    |                                                          |
    ========================================================= */

    if ( ! function_exists('mimimiRoot') ) {
        function mimimiRoot ( $toDisplay = TRUE ) {
            $value = mimimiServer('SCRIPT_NAME');
                $value = preg_replace('/[\s.]*[\/\\\\]+/u', '/', $value);
                $value = preg_replace('/\/+[^\/]+\.php$/ui', '/', $value);
                $value = preg_replace('/^\/+\s*/u', '', $value);
            if ($toDisplay) {
                echo $value;
            } else {
                return $value;
            }
        }
    }

    /* =========================================================
    |                                                          |
    |   Define a global routine to display or retrieve the     |
    |   page's relative URI                                    |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Input parameters:                                      |
    |       $toDisplay = BOOLEAN = TRUE if echo it to browser  |
    |                              FALSE if returns as result  |
    |                                                          |
    |   ----------------------------------------------------   |
    |                                                          |
    |   Output parameters:                                     |
    |       STRING = retrieved URI (without GET-parameters     |
    |                and leading or trailing slash)            |
    |                                                          |
    ========================================================= */

    if ( ! function_exists('mimimiUri') ) {
        function mimimiUri ( $toDisplay = TRUE ) {
            $encoding = 'UTF-8';
            $value = mimimiServer('REQUEST_URI');
                $value = preg_replace('/^([^\#?]*)[\#?].*$/u', '$1', $value);
                $value = preg_replace('/([\s.]*[\/\\\\])+/u', '/', $value);
                $value = preg_replace('/^(\/\s*)+/u', '', $value);
            $root = mimimiRoot(FALSE);
                $size = mb_strlen($root, $encoding);
                $test = mb_substr($value, 0, $size, $encoding);
                $test = mb_strtolower($test, $encoding) == mb_strtolower($root, $encoding);
                if ($test) {
                    $value = mb_substr($value, $size, 1000000, $encoding);
                }
                $value = preg_replace('/(\/[\s.]*)+$/u', '', $value);
            if ($toDisplay) {
                echo $value;
            } else {
                return $value;
            }
        }
    }
