<?php
/**
 * -------------------------------------------------------------------------
 *
 * STEP 1: As follows from the website logic proposed at the initial stage,
 *         your web application needs a module to work with a database table
 *         called "settings". That table will store a set of predefined
 *         settings, and this module will allow you to retrieve or edit
 *         their values. For this purpose, the initially empty PHP file
 *         below was created. And now you should connect the base table
 *         module using the "mimimiInclude" directive. And then you should
 *         declare a class "MyMimimiSettings" which extends the base table
 *         class. Next, you will need to follow STEPs 2-9 detailed below
 *         to program the required module.
 *
 * -------------------------------------------------------------------------
 *
 * @license  GPL-2.0  https://opensource.org/license/gpl-2-0/
 * @author   MiMiMi Community
 *
 * -------------------------------------------------------------------------
 */

mimimiInclude( 'ModuleWithTable.php' );
class MyMimimiSettings extends MimimiModuleWithTable {

    /**
     * ---------------------------------------------------------------------
     *
     * STEP 2: You need to specify a name of the database table to store
     *         website settings. It is quite logical to assume that the name
     *         of this table will correspond to its purpose, for example,
     *         "settings". Please note that due to MySQL naming restrictions,
     *         the maximum length of a table name is 64 characters. Do not
     *         exceed that length, otherwise you will receive an error message.
     *
     * ---------------------------------------------------------------------
     *
     * @access protected
     * @var    string
     *
     * ---------------------------------------------------------------------
     */

    protected $table = 'settings';

    /**
     * ---------------------------------------------------------------------
     *
     * STEP 3: You need to define a database table structure. Since your web
     *         application requires a simple settings schema, it makes sense
     *         to create a table with just three columns: a unique identifier,
     *         a unique settings alias, and a value. The identifier will be
     *         used to find the setting entry in the table when editing it.
     *         The alias will be used to look up the setting entry if its
     *         value needs to be displayed on screen or used in website
     *         logic. Please note that due to MySQL naming restrictions, the
     *         maximum length of a table column name is 64 characters.
     *
     * ---------------------------------------------------------------------
     *
     * @access protected
     * @var    array
     *
     * ---------------------------------------------------------------------
     */

    protected $tableFields = [
                  '`id`     BIGINT(20)    NOT NULL  AUTO_INCREMENT  COMMENT "unique setting identifier"',
                  '`alias`  VARCHAR(64)   NOT NULL                  COMMENT "unique alias to look up this setting"',
                  '`value`  VARCHAR(510)  NOT NULL                  COMMENT "value for this setting"'
              ];

    /**
     * ---------------------------------------------------------------------
     *
     * STEP 4: You need to define a list of table keys to speed up the
     *         database operations related to settings. According to the
     *         requirements of your web application, there are two main
     *         operations: update by ID, search by alias. So you only need
     *         to define the two keys below.
     *
     * ---------------------------------------------------------------------
     *
     * @access protected
     * @var    array
     *
     * ---------------------------------------------------------------------
     */

    protected $tableKeys = [
                  'PRIMARY KEY ( `id`    )',
                  'UNIQUE  KEY ( `alias` )'
              ];

    /**
     * ---------------------------------------------------------------------
     *
     * STEP 5: You should specify demo rows that will be used as default
     *         setting entries if the database does not have a table named
     *         "settings". In this case, all demo rows will be automatically
     *         added to the newly created table.
     *
     * ---------------------------------------------------------------------
     *
     * @access protected
     * @var    array
     *
     * ---------------------------------------------------------------------
     */

    protected $demoRows = [
        [ 'alias' => 'on_production',         'value' => '0'                           ],

        [ 'alias' => 'ips_editor',            'value' => '127.0.0.1, ::1, localhost'   ],
        [ 'alias' => 'ips_admin',             'value' => '127.0.0.1, ::1, localhost'   ],

        [ 'alias' => 'month_names',           'value' => 'January, February, March, April, May, June, July, August, September, October, November, December'                                                                                                                                                                                                  ],
        [ 'alias' => 'days_of_week',          'value' => 'Sunday, Monday, Tuesday, Wednesday, Thursday, Friday, Saturday'                                                                                                                                                                                                                                    ],
        [ 'alias' => 'languages',             'value' => 'ar, hy, az, be, bg, en, en-US, et, fi, fr, ka, de, it, kk, ky, lv, lt, mo, no, pl, pt, ro, ru, sr, sk, sl, es, sv, tg, th, tr, tk, uk, uz'                                                                                                                                                         ],
        [ 'alias' => 'post_sizes',            'value' => 'size1x1, size1x2, size1x3, size1x4, size1x5, size2x1, size2x2, size2x3, size2x4, size2x5, size3x1, size3x2, size3x3, size3x4, size3x5, size4x1, size4x2, size4x3, size4x4, size4x5, size5x1'                                                                                                       ],
        [ 'alias' => 'post_layouts',          'value' => 'layoutT-IC, layoutT-ICC, layoutT-ICCC, layoutT-ICCCC, layoutT-CI, layoutT-CCI, layoutT-CCCI, layoutT-CCCCI, layoutT-I-C, layoutT-I-CC, layoutT-I-CCC, layoutT-I-CCCC, layoutI-T-C, layoutI-T-CC, layoutI-T-CCC, layoutI-T-CCCC, layoutIT-C, layoutIT-CC, layoutIT-CCC, sidebarT-I-C, sidebarI-T-C' ],
        [ 'alias' => 'post_decors',           'value' => 'no-decor, border, borderT, borderTR, borderTRB, borderTRL, borderTB, borderTBL, borderTL, borderRB, borderRBL, borderRL, borderR, borderLB, borderB, borderL, contour, filled, coding'                                                                                                                       ],

        [ 'alias' => 'default_language',      'value' => 'en-US'                       ],
        [ 'alias' => 'default_edition',       'value' => 'The Demo Times'              ],
        [ 'alias' => 'default_credits_url',   'value' => ''                            ],
        [ 'alias' => 'default_number',        'value' => 'Issue №'                     ],
        [ 'alias' => 'default_slogan',        'value' => 'Single sheet edition'        ],
        [ 'alias' => 'default_sidebar',       'value' => 'Trends of last week'         ],
        [ 'alias' => 'default_copyright',     'value' => '© 2024 Demo Site'            ],

        [ 'alias' => 'button_home',           'value' => 'List Newspapers'             ],
        [ 'alias' => 'button_previous',       'value' => '« Prev'                      ],
        [ 'alias' => 'button_next',           'value' => 'Next »'                      ],
        [ 'alias' => 'button_latest',         'value' => 'Latest »'                    ],
        [ 'alias' => 'button_add',            'value' => 'Add Newspaper'               ],
        [ 'alias' => 'button_delete',         'value' => 'Delete'                      ],
        [ 'alias' => 'button_edit',           'value' => 'Edit'                        ],
        [ 'alias' => 'button_test',           'value' => 'Test This'                   ],
        [ 'alias' => 'button_settings',       'value' => 'Edit Settings'               ],
        [ 'alias' => 'button_moveleft',       'value' => 'Move to the left'            ],
        [ 'alias' => 'button_deletethis',     'value' => 'Mark this for deletion'      ],
        [ 'alias' => 'button_makenew',        'value' => 'Add a new one on the right'  ],
        [ 'alias' => 'button_moveright',      'value' => 'Move to the right'           ],

        [ 'alias' => 'home_language',         'value' => 'en-US'                       ],
        [ 'alias' => 'home_title',            'value' => 'Welcome to Demo Times'       ],
        [ 'alias' => 'home_meta',             'value' => 'Hi everyone! We are pleased to present you a lightweight website theme developed in March 2024. It\'s inspired by Olivia Ng\'s work on Codepen and focuses on blogs with a newspaper design. This beautiful theme is showcased on the current page, which shows how all its features work. Please select the newspaper issue below that you would like to read now. Or just click the "Latest »" button to read the last newspaper issue.' ],
        [ 'alias' => 'home_edition',          'value' => 'The Demo Times'              ],
        [ 'alias' => 'home_number',           'value' => 'Hello to You'                ],
        [ 'alias' => 'home_slogan',           'value' => 'Select the issue below'      ],
        [ 'alias' => 'home_copyright',        'value' => '© 2024 Demo Site'            ],

        [ 'alias' => '404_language',          'value' => 'en-US'                       ],
        [ 'alias' => '404_title',             'value' => 'Page is not found!'          ],
        [ 'alias' => '404_meta',              'value' => 'Oops, something went wrong! There is no such page on our website. Please check that the URL you entered is correct. The most likely causes are: you have used an outdated link, or you have used an incorrect link, or the address of the page is typed incorrectly, or the page is closed from unauthorized users.' ],
        [ 'alias' => '404_edition',           'value' => 'Page Not Found'              ],
        [ 'alias' => '404_number',            'value' => 'Home » Error 404'            ],
        [ 'alias' => '404_slogan',            'value' => 'Something went wrong!'       ],

        [ 'alias' => 'add_language',          'value' => 'en-US'                       ],
        [ 'alias' => 'add_title',             'value' => 'Add Newspaper'               ],
        [ 'alias' => 'add_meta',              'value' => 'Please complete the form below to add a newspaper issue to your website.' ],
        [ 'alias' => 'add_edition',           'value' => 'Add Newspaper'               ],
        [ 'alias' => 'add_number',            'value' => 'Admin » Newspapers » Add'    ],
        [ 'alias' => 'add_slogan',            'value' => 'Complete this form'          ],

        [ 'alias' => 'delete_language',       'value' => 'en-US'                       ],
        [ 'alias' => 'delete_title',          'value' => 'Delete Newspaper'            ],
        [ 'alias' => 'delete_meta',           'value' => 'Are you sure you want to remove this newspaper issue from your website? To perform this action, you need to check the "Yes, I am sure" box below and click the "Delete" button.' ],
        [ 'alias' => 'delete_edition',        'value' => 'Delete Newspaper'            ],
        [ 'alias' => 'delete_number',         'value' => 'Admin » Newspapers » Delete' ],
        [ 'alias' => 'delete_slogan',         'value' => 'Confirm this action'         ],
        [ 'alias' => 'delete_sure',           'value' => 'Yes, I am sure'              ],
        [ 'alias' => 'delete_submit',         'value' => 'Delete'                      ],
        [ 'alias' => 'delete_cancel',         'value' => 'Cancel'                      ],

        [ 'alias' => 'edit_language',         'value' => 'en-US'                       ],
        [ 'alias' => 'edit_title',            'value' => 'Edit Newspaper'              ],
        [ 'alias' => 'edit_meta',             'value' => 'Please correct the information in the form below in order to edit this newspaper issue on your website.' ],
        [ 'alias' => 'edit_edition',          'value' => 'Edit Newspaper'              ],
        [ 'alias' => 'edit_number',           'value' => 'Admin » Newspapers » Edit'   ],
        [ 'alias' => 'edit_slogan',           'value' => 'Correct form below'          ],
        [ 'alias' => 'edit_submit',           'value' => 'Save'                        ],
        [ 'alias' => 'edit_cancel',           'value' => 'Cancel'                      ],

        [ 'alias' => 'settings_language',     'value' => 'en-US'                       ],
        [ 'alias' => 'settings_title',        'value' => 'Edit Your Site Settings'     ],
        [ 'alias' => 'settings_edition',      'value' => 'Site Settings'               ],
        [ 'alias' => 'settings_number',       'value' => 'Admin » Settings » Edit'     ],
        [ 'alias' => 'settings_slogan',       'value' => 'Correct rows below'          ],
        [ 'alias' => 'settings_submit',       'value' => 'Save'                        ],
        [ 'alias' => 'settings_cancel',       'value' => 'Cancel'                      ],

        [ 'alias' => 'label_enabled',         'value' => 'Visible For All'             ],
        [ 'alias' => 'label_language',        'value' => 'Language Code:'              ],
        [ 'alias' => 'label_url',             'value' => 'Newspaper URL:'              ],
        [ 'alias' => 'label_edition',         'value' => 'Newspaper Name:'             ],
        [ 'alias' => 'label_credits_url',     'value' => 'Newspaper Credits Link:'     ],
        [ 'alias' => 'label_slogan',          'value' => 'Issue Slogan:'               ],
        [ 'alias' => 'label_number',          'value' => 'Issue Number:'               ],
        [ 'alias' => 'label_date',            'value' => 'Issue Date:'                 ],
        [ 'alias' => 'label_meta',            'value' => 'Issue Summary Text:'         ],
        [ 'alias' => 'label_sidebar',         'value' => 'Side Bar Title:'             ],
            [ 'alias' => 'label_size',        'value' => 'Article Box Size:'           ],
            [ 'alias' => 'label_layout',      'value' => 'Article Box Layout:'         ],
            [ 'alias' => 'label_decor',       'value' => 'Article Box Decoration:'     ],
            [ 'alias' => 'label_h1',          'value' => 'Article Title:'              ],
            [ 'alias' => 'label_text',        'value' => 'Article Text:'               ],
            [ 'alias' => 'label_image',       'value' => 'Image URL:'                  ],
            [ 'alias' => 'label_img_file',    'value' => 'Image To Upload:'            ],
            [ 'alias' => 'label_image_alt',   'value' => 'Image Label:'                ],
            [ 'alias' => 'label_backlink',    'value' => 'Link To Source:'             ],
        [ 'alias' => 'label_copyright',       'value' => 'Newspaper Copyright:'        ],

        [ 'alias' => 'remark_language',       'value' => ''                            ],
        [ 'alias' => 'remark_url',            'value' => 'Leave this field blank if you want to automatically generate the page URL. In this case, it will consist of a random set of characters.' ],
        [ 'alias' => 'remark_edition',        'value' => 'This field is also the value of the <title> tag in document headers.' ],
        [ 'alias' => 'remark_credits_url',    'value' => 'Leave this field blank if you do not want the backlink to be in the footer.' ],
        [ 'alias' => 'remark_slogan',         'value' => ''                            ],
        [ 'alias' => 'remark_number',         'value' => ''                            ],
        [ 'alias' => 'remark_date',           'value' => ''                            ],
        [ 'alias' => 'remark_meta',           'value' => 'This field is also used as a short description in the list of newspapers on the home page.' ],
        [ 'alias' => 'remark_sidebar',        'value' => ''                            ],
            [ 'alias' => 'remark_size',       'value' => ''                            ],
            [ 'alias' => 'remark_layout',     'value' => ''                            ],
            [ 'alias' => 'remark_decor',      'value' => ''                            ],
            [ 'alias' => 'remark_h1',         'value' => ''                            ],
            [ 'alias' => 'remark_text',       'value' => ''                            ],
            [ 'alias' => 'remark_image',      'value' => ''                            ],
            [ 'alias' => 'remark_img_file',   'value' => ''                            ],
            [ 'alias' => 'remark_image_alt',  'value' => ''                            ],
            [ 'alias' => 'remark_backlink',   'value' => 'Leave this field blank if you do not want the backlink to be on the title of this article or on its image when there is no title.' ],
        [ 'alias' => 'remark_copyright',      'value' => ''                            ],

        [ 'alias' => 'marker_moderating',     'value' => 'It is not visible now!'      ],
        [ 'alias' => 'error_sure',            'value' => 'Please confirm your wish by checking the "Yes, I am sure" box below.' ],
        [ 'alias' => 'error_id_spoofing',     'value' => 'Your request cannot be accepted because an attempt to spoof the ID field has been detected.' ],
        [ 'alias' => 'error_empty_newspaper', 'value' => 'You must fill in all the fields, or at least these: Newspaper Name, Issue Slogan, Issue Number, Issue Date, Newspaper Copyright.' ],
        [ 'alias' => 'error_no_articles',     'value' => 'You must complete at least one of the sections of your newspaper: Article Title, Article Text, Image URL or Image To Upload.' ],
        [ 'alias' => 'error_url_conflict',    'value' => 'The newspaper URL you entered has been conflicted with another page.' ],
        [ 'alias' => 'error_bad_image',       'value' => 'The uploaded image file must be in WEBP, JPEG, PNG, SVG, GIF or BMP format.' ],
        [ 'alias' => 'error_demo',            'value' => 'Everything is fine! However, this website did not save your changes because you are currently in demo mode. In this mode, any addition/change/deletion of pages is simply simulated to show you how it would work on a real site.' ]
    ];

    /**
     * ---------------------------------------------------------------------
     *
     * STEP 6: You should declare and implement a public method to look up
     *         a setting entry by its alias and return its decoded value.
     *         Please note that the decoded value is under the index "decoded".
     *         Also note that this method uses two other local methods:
     *         "decodeItem" and "getItem". You will implement them below in
     *         STEPs 7 and 8.
     *
     * ---------------------------------------------------------------------
     *
     * @public
     * @param   string  $alias  The alias for the setting you are looking for.
     * @return  mixed           The setting value that was obtained.
     *
     * ---------------------------------------------------------------------
     */

    public function getValue ( $alias ) {
        $item = $this->decodeItem(
                    $this->getItem( $alias )
                );
        return $item ? $item[ 'decoded' ]
                     : '';
    }

    /**
     * ---------------------------------------------------------------------
     *
     * STEP 7: You should declare and implement a public method to decode
     *         settings entries if they have certain values that need to be
     *         decoded before they can be used. For example, the setting
     *         "on_production" is a bit flag, so you need to decode it as
     *         a boolean value. A setting like "ips_editor" is a comma
     *         separated list of IP addresses, so you need to decode it as
     *         an array. Please note that the decoded value must be stored
     *         under the index "decoded".
     *
     * ---------------------------------------------------------------------
     *
     * @public
     * @param   array  $item  The entry that need to be decoded.
     * @return  array         Decoded entry.
     *
     * ---------------------------------------------------------------------
     */

    public function decodeItem ( $item ) {
        if ( isset( $item[ 'alias' ] ) ) {
            switch ( $item[ 'alias' ] ) {

                // these are bit flags
                case 'on_production':
                    $item[ 'decoded' ] = ! empty( $item[ 'value' ] );
                    break;

                // these are lists
                case 'ips_editor':
                case 'ips_admin':
                case 'month_names':
                case 'days_of_week':
                case 'languages':
                case 'post_sizes':
                case 'post_layouts':
                case 'post_decors':
                    $comma     = '[\s,]+';
                    $trails    = '~(^' . $comma . '|' . $comma . '$)~u';
                    $separator = '~'   . $comma                  . '~u';
                    $item[ 'decoded' ] = preg_replace( $trails,    '', $item[ 'value' ]   );
                    $item[ 'decoded' ] = preg_split  ( $separator,     $item[ 'decoded' ] );
                    break;

                // otherwise, no decoding
                default:
                    $item[ 'decoded' ] = $item[ 'value' ];
            }
        }
        return $item;
    }

    /**
     * ---------------------------------------------------------------------
     *
     * STEP 8: You should declare and implement a public method to retrieve
     *         a setting entry by its alias. Please note that "t1" below is
     *         an alias for the "settings" database table.
     *
     * ---------------------------------------------------------------------
     *
     * @public
     * @param   string      $alias  The alias for the setting you are looking for.
     * @return  array|bool          ARRAY on success. It contains a row obtained from a database table.
     *                              FALSE on failure. This means the entry you are looking for was not found.
     *
     * ---------------------------------------------------------------------
     */

    public function getItem ( $alias ) {

        // build a filter by alias
        $filter = [
            't1.alias' => $alias
        ];

        // find the entry
        return $this->select( $filter );
    }

    /**
     * ---------------------------------------------------------------------
     *
     * STEP 9: Finally, you should declare and implement a public method to
     *         get all settings entries, sorted alphabetically by the
     *         setting alias. Please note that "t1" below is an alias for
     *         the "settings" database table.
     *
     * ---------------------------------------------------------------------
     *
     * @public
     * @return  array|bool  ARRAY on success. Each element is an array, like a row obtained from a database table.
     *                      FALSE on failure. This means no entries were found.
     *
     * ---------------------------------------------------------------------
     */

    public function getItems () {

        // build a sorter by alias
        $filter = [
            'orderby' => [
                't1.alias' => 'asc'
            ]
        ];

        // find entries
        return $this->select( $filter, 0, 1000000000 );
    }
};

/**
 * -------------------------------------------------------------------------
 *
 * Everything is very good! You should now go to STAGE 4 to examine the file
 * "newspaper/Newspapers/Newspapers.php" that implements this stage.
 *
 * -------------------------------------------------------------------------
 */