<?php
/**
 * -------------------------------------------------------------------------
 * The SETTINGS module to work with the SETTINGS database table.
 *
 * This module is loaded automatically. It fires the moment a web developer
 * first touches this module in the source code of the website's backend or
 * frontend.
 *
 * -------------------------------------------------------------------------
 *
 * Overridden properties below are:
 *     PROTECTED  $table  ───────────> PUBLIC
 *     PROTECTED  $tableFields
 *                      ├──> setting_id           INTEGER (auto incremented)
 *                      ├──> settings_section_id  INTEGER
 *                      ├──> setting_alias        STRING  (length 50)
 *                      ├──> setting_name         STRING  (length 150)
 *                      ├──> setting_info         STRING  (length 500)
 *                      ├──> setting_value        STRING  (length 5000)
 *                      ├──> setting_order        INTEGER
 *                      └──> setting_modified     TIMESTAMP
 *     PROTECTED  $tableKeys
 *                      ├──> setting_id                          UNIQUE
 *                      ├──> settings_section_id, setting_alias  UNIQUE
 *                      ├──> settings_section_id, setting_name   UNIQUE
 *                      ├──> settings_section_id
 *                      ├──> setting_alias
 *                      ├──> setting_name
 *                      ├──> settings_section_id, setting_order, setting_alias
 *                      ├──> settings_section_id, setting_order, setting_name
 *                      ├──> setting_order, setting_alias
 *                      ├──> setting_order, setting_name
 *                      └──> setting_modified
 *     PROTECTED  $myUmbrellaFile
 *
 * Overridden methods below are:
 *     save
 *
 * Implemented methods below are:
 *     decodeItem
 *     getItem
 *     getItems
 *     getDemo
 *     getValue
 *
 * -------------------------------------------------------------------------
 *
 * @package    MimimiFramework
 * @subpackage Modules
 * @copyright  2022 MiMiMi Community
 *             https://mimimi.software/
 * @license    CC BY 4
 *             https://www.creativecommons.org/licenses/by/4.0
 * -------------------------------------------------------------------------
 */

mimimiInclude('UmbrellaWithTable.php');
class MimimiSettings extends MimimiUmbrellaModuleWithTable {

    /**
     * ---------------------------------------------------------------------
     * Database table name.
     *
     * Note: The maximum length for table names, column names, and indexes
     *       is 64 characters.
     *
     * @var string
     * @access public
     * ---------------------------------------------------------------------
     */

    public $table = 'settings';

    /**
     * ---------------------------------------------------------------------
     * Database table columns.
     *
     * Note: The maximum length for table names, column names, and indexes
     *       is 64 characters.
     *
     * Note: Data type storage requirements:
     *       BIGINT    = 8 bytes;
     *       VARCHAR   = the actual length of a given string value + 1
     *                   byte if the multibyte value column requires 0 to
     *                   255 bytes to store the values, or the actual length
     *                   + 2 bytes if that column requires more than 255
     *                   bytes;
     *       TIMESTAMP = 4 bytes + fractional seconds storage (from 0 to 3
     *                   bytes, depending on the fractional seconds
     *                   precision of stored values).
     *
     * Note: To calculate the actual number of bytes used to store a
     *       particular VARCHAR column value, you must take into account
     *       the character set used for that column and whether the value
     *       contains multibyte characters. In particular, when using
     *       a UTF-8 Unicode character set, you must keep in mind that not
     *       all characters use the same number of bytes. "utf8mb3" and
     *       "utf8mb4" character sets can require up to three and four
     *       bytes per character, respectively. For example, a VARCHAR(255)
     *       column can hold a string with a maximum length of 255
     *       characters. Assuming that the column uses the "latin1"
     *       character set (one byte per character), the actual storage
     *       required is the length of the string, plus one byte to record
     *       the length of the string. For the string "abcd", its length
     *       is 4 and the storage requirement is five bytes. If the same
     *       column is instead declared to use the "ucs2" double-byte
     *       character set, the storage requirement is 10 bytes: The
     *       length of "abcd" is eight bytes and the column requires two
     *       bytes to store lengths because the maximum length is greater
     *       than 255 (up to 510 bytes).
     *
     * Note: The effective maximum number of bytes that can be stored in
     *       a VARCHAR column is subject to the maximum row size of 65535
     *       bytes, which is shared among all columns. For a VARCHAR column
     *       that stores multibyte characters, the effective maximum number
     *       of characters is less. For example, "utf8mb4" characters can
     *       require up to four bytes per character, so a VARCHAR column
     *       that uses the "utf8mb4" character set can be declared to be
     *       a maximum of 16383 characters.
     *
     * Note: The internal representation of a MySQL table has a maximum row
     *        size limit of 65535 bytes, not counting BLOB columns, even if
     *        the storage engine is capable of supporting larger rows.
     *        Otherwise you will get an error when running your MySQL query.
     *
     * @var array
     * @access protected
     * ---------------------------------------------------------------------
     */

    protected $tableFields = [
                  '`setting_id`           BIGINT(20)     NOT NULL  AUTO_INCREMENT                             COMMENT "unique setting identifier"',
                  '`settings_section_id`  BIGINT(20)     NOT NULL                                             COMMENT "identifier of the settings section"',
                  '`setting_alias`        VARCHAR(50)    NOT NULL  DEFAULT ""                                 COMMENT "pairwise unique alias to look for this setting"',
                  '`setting_name`         VARCHAR(150)   NOT NULL  DEFAULT ""                                 COMMENT "pairwise unique name for this setting to display"',
                  '`setting_info`         VARCHAR(500)   NOT NULL  DEFAULT ""                                 COMMENT "optional description of this setting if you need to explain it"',
                  '`setting_value`        VARCHAR(5000)  NOT NULL  DEFAULT ""                                 COMMENT "value for this setting"',
                  '`setting_order`        INT(11)        NOT NULL                                             COMMENT "position of this setting when sorting within its section"',
                  '`setting_modified`     TIMESTAMP      NULL      DEFAULT NULL  ON UPDATE CURRENT_TIMESTAMP  COMMENT "last time this setting was changed"'
              ];

    /**
     * ---------------------------------------------------------------------
     * Database table keys.
     *
     * Note: The maximum length for table names, column names, and indexes
     *       is 64 characters.
     *
     * @var array
     * @access protected
     * ---------------------------------------------------------------------
     */

    protected $tableKeys = [
                  'PRIMARY KEY (`setting_id`)',
                  'KEY         (`settings_section_id`)',
                  'UNIQUE  KEY (`settings_section_id`, `setting_alias`)',
                  'KEY         (                       `setting_alias`)',
                  'UNIQUE  KEY (`settings_section_id`, `setting_name`)',
                  'KEY         (                       `setting_name`)',
                  'KEY         (`settings_section_id`, `setting_order`, `setting_alias`)',
                  'KEY         (                       `setting_order`, `setting_alias`)',
                  'KEY         (`settings_section_id`, `setting_order`, `setting_name`)',
                  'KEY         (                       `setting_order`, `setting_name`)',
                  'KEY         (`setting_modified`)'
              ];

    /**
     * ---------------------------------------------------------------------
     * Full filename of this script file.
     *
     * See the "mimimi.core/UmbrellaWithTable.php" file to understand this
     * technical property. According to the framework standard, this property
     * must be reinitialized in the same way in any new umbrella type module.
     * Therefore, the same line is repeated here as in the file just mentioned.
     *
     * @var string
     * @access protected
     * ---------------------------------------------------------------------
     */

    protected $myUmbrellaFile = __FILE__;

    /**
     * ---------------------------------------------------------------------
     * Saves entry of the setting in the database.
     *
     * @public
     * @param   array     $item      The list of fields that need to be saved for the entry.
     * @param   string    $idColumn  (optional) The name of the column containing the entry identifier.
     * @return  int|bool             INTEGER if success, it is identifier of the saved entry.
     *                               FALSE   if failure.
     * ---------------------------------------------------------------------
     */

    public function save ( $item, $idColumn = 'setting_id' ) {
        $this->sections->install();

        // save fields corresponding to the settings section
        $id = $this->sections->save($item);
        if ( $id ) {
            $item['settings_section_id'] = $id;
        }

        // save fields corresponding to the setting
        return parent::save($item, $idColumn);
    }

    /**
     * ---------------------------------------------------------------------
     * Decodes the setting entry.
     *
     * Note: You will have to override this method later if your settings
     *       have certain values that need to be decoded before using them.
     *
     * @public
     * @param   array  $item  The list of fields that need to be decoded for the entry.
     * @return  array         Decoded list of fields.
     * ---------------------------------------------------------------------
     */

    public function decodeItem ( $item ) {
        return $item;
    }

    /**
     * ---------------------------------------------------------------------
     * Retrieves the setting entry by its alias.
     *
     * Note: "t1" below is an alias for the "settings" table, "t2" is an
     *       alias for the "sections" table.
     *
     * @public
     * @param   string      $alias    The setting alias you are looking for.
     * @param   int|string  $section  (optional) if INTEGER, then it's the identifier of the settings section being filtered.
     *                                           if STRING,  then it's the alias of the settings section being filtered.
     * @return  array|bool            ARRAY on success, it contains a database table row.
     *                                FALSE if that entry is not found.
     * ---------------------------------------------------------------------
     */

    public function getItem ( $alias, $section = null ) {
        $filter = [

            // select all columns
            'select' => [
                't1.*' => TRUE,
                't2.*' => TRUE
            ],

            // attach the "sections" table by ID
            'join' => [
                $this->sections->table => [
                    't2.settings_section_id' => 't1.settings_section_id'
                ]
            ],

            // conditions for the WHERE clause
            't1.setting_alias' => $alias
        ];

        // filter by section if one is specified
        if ( is_numeric($section) ) {
            $filter['t1.settings_section_id']    = $section;
        } else if ( is_string($section) ) {
            $filter['t2.settings_section_alias'] = $section;

        // otherwise filter only the oldest entry
        // (in case another setting from different
        // section have the same name, and we are
        // searching only by name without specifying
        // the section)
        } else {
            $filter['orderby'] = [
                't1.setting_id' => 'asc'
            ];
        }

        // find entry
        return $this->select($filter);
    }

    /**
     * ---------------------------------------------------------------------
     * Retrieves the settings entries listed on the numbered page.
     *
     * Note: "t1" below is an alias for the "settings" table, "t2" is an
     *       alias for the "sections" table.
     *
     * @public
     * @param   int         $page     The requested page number.
     * @param   int         $size     (optional) The capacity of the page. Default size is 24 (that is 2 * 3 * 4 entries).
     * @param   int|string  $section  (optional) if INTEGER, then it's the identifier of the settings section being filtered.
     *                                           if STRING,  then it's the alias of the settings section being filtered.
     * @return  array|bool            ARRAY on success, each item is an array like a database table row.
     *                                FALSE if entries are not found.
     * ---------------------------------------------------------------------
     */

    public function getItems ( $page, $size = 24, $section = null ) {
        $filter = [

            // select all columns
            'select' => [
                't1.*' => TRUE,
                't2.*' => TRUE
            ],

            // attach the "sections" table by ID
            'join' => [
                $this->sections->table => [
                    't2.settings_section_id' => 't1.settings_section_id'
                ]
            ],

            // sort by setting position and name
            'orderby' => [
                't1.setting_order' => 'asc',
                't1.setting_name'  => 'asc'
            ]
        ];

        // filter by section if one is specified
        if ( is_numeric($section) ) {
            $filter['t1.settings_section_id']    = $section;
        } else if ( is_string($section) ) {
            $filter['t2.settings_section_alias'] = $section;

        // otherwise sort by section position and name then setting position and name
        } else {
            $filter['orderby'] = [
                't2.settings_section_order' => 'asc',
                't2.settings_section_name'  => 'asc',
                't1.setting_order'          => 'asc',
                't1.setting_name'           => 'asc'
            ];
        }

        // find entries
        $items = $this->select($filter, ($page - 1) * $size, $size + 1);

        // make a NextPage marker
        if ( $items ) {
            if ( count($items) > $size ) {
                array_pop($items);
                $items[] = FALSE;
            }
        }

        return $items;
    }

    /**
     * ---------------------------------------------------------------------
     * Finds the demo value of a named setting.
     *
     * @public
     * @param   string  $alias    The setting alias you are looking for.
     * @param   int     $section  (optional) The identifier of the settings section being filtered.
     * @param   mixed   $default  (optional) The default value if that demo setting not found.
     * @return  mixed             Retrieved value.
     * ---------------------------------------------------------------------
     */

    public function getDemo ( $alias, $section = null, $default = '' ) {
        if ( $this->demoRows ) {
            foreach ( $this->demoRows as $item ) {

                // filter by setting alias
                if ( isset($item['setting_alias']) ) {
                    if ( $alias == $item['setting_alias'] ) {

                        // filter by section if one is specified
                        if ( is_null($section)
                             || isset($item['settings_section_id'])
                                && is_numeric($section)
                                   && $section == $item['settings_section_id']
                             || isset($item['settings_section_alias'])
                                && is_string($section)
                                   && $section == $item['settings_section_alias'] ) {

                            // retrieve the value
                            $item = $this->decodeItem($item);
                            return isset($item['setting_value'])
                                   ? $item['setting_value']
                                   : $default;
                        }
                    }
                }
            }
        }
        return $default;
    }

    /**
     * ---------------------------------------------------------------------
     * Finds the real value of a named setting.
     *
     * @public
     * @param   string      $alias    The setting alias you are looking for.
     * @param   int|string  $section  (optional) if INTEGER, then it's the identifier of the settings section being filtered.
     *                                           if STRING,  then it's the alias of the settings section being filtered.
     * @param   mixed       $default  (optional) The default value if that real setting not found.
     * @return  mixed                 Retrieved value.
     * ---------------------------------------------------------------------
     */

    public function getValue ( $alias, $section = null, $default = '' ) {

        // retrieve the setting entry
        $item = $this->getItem($alias, $section);

        // retrieve the value
        if ( $item ) {
            $item = $this->decodeItem($item);
            return isset($item['setting_value'])
                   ? $item['setting_value']
                   : $default;
        }
        return $this->getDemo($alias, $section, $default);
    }
};