/* =============================================================
|                                                              |
|   ErrorPanel v.1.0                                           |
|       Author: MiMiMi Community                               |
|           > www.mimimi.software                              |
|       Licensed under CC BY 4                                 |
|           > www.creativecommons.org/licenses/by/4.0          |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   This file is used to display messages in the error panel   |
|   that slides across the bottom of your website.             |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   For example, let's add a panel on the website page:        |
|                                                              |
|       <aside class="error-panel"></aside>                    |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   For example, let's add this file after website footer:     |
|                                                              |
|       <script src="<?php theme() ?>js/error-panel.js">       |
|       </script>                                              |
|       <script>                                               |
|           cms.errors = new ErrorPanelClass();                |
|       </script>                                              |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   For example, let's display a message:                      |
|                                                              |
|       <script>                                               |
|           cms.errors.display('Hello World!');                |
|       </script>                                              |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   For example, let's clear this panel:                       |
|                                                              |
|       <script>                                               |
|           cms.errors.display();                              |
|       </script>                                              |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   If you want to use the ErrorPanel component in             |
|   a specific website theme, for example YOUR-THEME-NAME,     |
|   then this file must be placed in the appropriate folder    |
|   of that theme.                                             |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   Note: This file is part of the ErrorPanel component        |
|         that consists these files:                           |
|             themes/YOUR-THEME-NAME/css/error-panel.css       |
|             themes/YOUR-THEME-NAME/js/error-panel.js         |
|                                                              |
|   --------------------------------------------------------   |
|                                                              |
|   Note: If you click on the panel, it will slide out and     |
|         vice versa. If you double click on the panel, it     |
|         will hide.                                           |
|                                                              |
============================================================= */

    function ErrorPanelClass ( selector = '.error-panel' ) {

        /* =====================================================
        |                                                      |
        |   Declare the private properties                     |
        |                                                      |
        ===================================================== */

        var me            = this,
            flagActive    = 'data-ItIsActiveNow',
            timer         = 0,
            panelLifetime = 5000,
            panelNode     = false,
            panelSelector = selector;

        /* =====================================================
        |                                                      |
        |   Declare the public name of this component          |
        |                                                      |
        ===================================================== */

        me.name = 'ErrorPanel';

        /* =====================================================
        |                                                      |
        |   Stop an event propagation                          |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       event = OBJECT = user's click info             |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        function stopEvent ( event ) {
            event.stopImmediatePropagation();
            event.preventDefault();
        };

        /* =====================================================
        |                                                      |
        |   Look up the node of the error panel                |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       NONE                                           |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        function findPanel () {
            if (panelNode === false) {
                panelNode = document.querySelector(panelSelector);
                if (panelNode) {
                    with (panelNode) {
                        addEventListener('dblclick', clearPanel,  true);
                        addEventListener('click',    togglePanel, true);
                    }
                }
            }
        };

        /* =====================================================
        |                                                      |
        |   Clear the panel                                    |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       event = OBJECT = user's click info             |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        function clearPanel ( event ) {
            stopEvent(event);
            me.display();
        };

        /* =====================================================
        |                                                      |
        |   Switch the panel state                             |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       event = OBJECT = user's click info             |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        function togglePanel ( event ) {
            stopEvent(event);
            if (panelNode.hasAttribute(flagActive)) {
                hidePanel();
            } else {
                showPanel();
            }
        };

        /* =====================================================
        |                                                      |
        |   Slide in the panel                                 |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       NONE                                           |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        function hidePanel () {
            if (timer) {
                clearTimeout(timer);
                timer = 0;
            }
            panelNode.removeAttribute(flagActive);
        };

        /* =====================================================
        |                                                      |
        |   Slide out the panel                                |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       NONE                                           |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        function showPanel () {
            if (panelNode.innerText != '') {
                panelNode.setAttribute(flagActive, true);
                timer = setTimeout(hidePanel, panelLifetime);
            }
        };

        /* =====================================================
        |                                                      |
        |   Display a text message on the panel                |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Input parameters:                                  |
        |       message = STRING = text to display             |
        |                                                      |
        |   ------------------------------------------------   |
        |                                                      |
        |   Output parameters:                                 |
        |       NONE                                           |
        |                                                      |
        ===================================================== */

        me.display = function ( message = '' ) {
            findPanel();
            if (panelNode) {
                hidePanel();
                panelNode.innerText = message;
                showPanel();
            }
        };
    };
